/*******************************************************************************
 * Copyright (c) 2000, 2007 IBM Corporation and others.
 * Portion Copyright (c) 2008 Nokia Corporation and/or its subsidiary(-ies).
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *     Nokia Corporation - Qt implementation
 *******************************************************************************/
package org.eclipse.swt.graphics;


import org.eclipse.swt.*;
import org.eclipse.swt.internal.qt.OS;
import org.eclipse.swt.internal.qt.QtSupplementaryFontData;





/**
 * Instances of this class describe operating system fonts.
 * <p>
 * For platform-independent behaviour, use the get and set methods
 * corresponding to the following properties:
 * <dl>
 * <dt>height</dt><dd>the height of the font in points</dd>
 * <dt>name</dt><dd>the face name of the font, which may include the foundry</dd>
 * <dt>style</dt><dd>A bitwise combination of NORMAL, ITALIC and BOLD</dd>
 * </dl>
 * If extra, platform-dependent functionality is required:
 * <ul>
 * <li>On <em>Windows</em>, the data member of the <code>FontData</code>
 * corresponds to a Windows <code>LOGFONT</code> structure whose fields
 * may be retrieved and modified.</li>
 * <li>On <em>X</em>, the fields of the <code>FontData</code> correspond
 * to the entries in the font's XLFD name and may be retrieved and modified.
 * </ul>
 * Application code does <em>not</em> need to explicitly release the
 * resources managed by each instance when those instances are no longer
 * required, and thus no <code>dispose()</code> method is provided.
 *
 * @see Font
 */
public final class FontData {
	/**
	 * the font name
	 */
	String name;

	/**
	 * The height of the font data in points
	 */
	int height;

	/**
	 * the font style
	 */
	int style;

	/**
	 * The locales of the font
	 */
	String lang, country, variant;

	QtSupplementaryFontData extraFontData;
	String xlfd;





/**
 * Constructs a new uninitialized font data.
 */
public FontData () {
	this("", Device.FONT_DEF_HEIGHT, SWT.NORMAL);
}

//The characters in the string must all be decimal digits
// the value must be given with radix 10

/**
 * Constructs a new FontData given a string representation
 * in the form generated by the <code>FontData.toString</code>
 * method.
 * <p>
 * Note that the representation varies between platforms,
 * and a FontData can only be created from a string that was
 * generated on the same platform.
 * </p>
 *
 * @param string the string representation of a <code>FontData</code> (must not be null)
 *
 * @exception IllegalArgumentException <ul>
 *    <li>ERROR_NULL_ARGUMENT - if the argument is null</li>
 *    <li>ERROR_INVALID_ARGUMENT - if the argument does not represent a valid description</li>
 * </ul>
 *
 * @see #toString
 */
public FontData(String string) {
	if (string == null) SWT.error(SWT.ERROR_NULL_ARGUMENT);
	parseFontDescriptor(string);
	updateFontData();

}

/**
 * Constructs a new font data given a font name,
 * the height of the desired font in points,
 * and a font style.
 *
 * @param name the name of the font (must not be null)
 * @param height the font height in points
 * @param style a bit or combination of NORMAL, BOLD, ITALIC
 *
 * @exception IllegalArgumentException <ul>
 *    <li>ERROR_NULL_ARGUMENT - when the font name is null</li>
 *    <li>ERROR_INVALID_ARGUMENT - if the height is negative</li>
 * </ul>
 */
public FontData(String name, int height, int style) {
	if (name == null) SWT.error(SWT.ERROR_NULL_ARGUMENT);
	if (height < 0) SWT.error(SWT.ERROR_INVALID_ARGUMENT);
	setName(name);
	setHeight(height);
	setStyle(style);
}

/**
 * Compares the argument to the receiver, and returns true
 * if they represent the <em>same</em> object using a class
 * specific comparison.
 *
 * @param object the object to compare with this object
 * @return <code>true</code> if the object is the same as this object and <code>false</code> otherwise
 *
 * @see #hashCode
 */
	public boolean equals(Object object) {
		if (object == this)
			return true;
		if (!(object instanceof FontData))
			return false;

		FontData data = (FontData) object;

		if (!((xlfd == null && data.xlfd == null) || (xlfd != null && data.xlfd != null)))
			return false;

		if (!((extraFontData == null && data.extraFontData == null) || (extraFontData != null && data.extraFontData != null)))
			return false;

		if (xlfd != null) {
			return xlfd.trim().equals(data.xlfd.trim());
		}

		boolean mainFontStyleEqual = name.equals(data.name)
				&& height == data.height && style == data.style;
		if (mainFontStyleEqual && extraFontData != null) {
			return mainFontStyleEqual
					&& extraFontData.underline == data.extraFontData.underline
					&& extraFontData.overline == data.extraFontData.overline
					&& extraFontData.strikeOut == data.extraFontData.strikeOut
					&& extraFontData.stretch == data.extraFontData.stretch
					&& extraFontData.fixedPitch == data.extraFontData.fixedPitch
					&& extraFontData.style == data.extraFontData.style
					&& extraFontData.weight == data.extraFontData.weight
					&& extraFontData.styleStrategy == data.extraFontData.styleStrategy;
		} else {
			return mainFontStyleEqual;
		}
	}

/**
 * Returns the height of the receiver in points.
 *
 * @return the height of this FontData
 *
 * @see #setHeight(int)
 */
public int getHeight() {
	return height;
}

/**
 * Returns the locale of the receiver.
 * <p>
 * The locale determines which platform character set this
 * font is going to use. Widgets and graphics operations that
 * use this font will convert UNICODE strings to the platform
 * character set of the specified locale.
 * </p>
 * <p>
 * On platforms where there are multiple character sets for a
 * given language/country locale, the variant portion of the
 * locale will determine the character set.
 * </p>
 *
 * @return the <code>String</code> representing a Locale object
 * @since 3.0
 */
public String getLocale () {
	StringBuffer buffer = new StringBuffer ();
	char sep = '_';
	if (lang != null) {
		buffer.append (lang);
		buffer.append (sep);
	}
	if (country != null) {
		buffer.append (country);
		buffer.append (sep);
	}
	if (variant != null) {
		buffer.append (variant);
	}

	String result = buffer.toString ();
	int length = result.length ();
	if (length > 0) {
		if (result.charAt (length - 1) == sep) {
			result = result.substring (0, length - 1);
		}
	}
	return result;
}

/**
 * Returns the name of the receiver.
 * On platforms that support font foundries, the return value will
 * be the foundry followed by a dash ("-") followed by the face name.
 *
 * @return the name of this <code>FontData</code>
 *
 * @see #setName
 */
public String getName() {
	return name;
}

/**
 * Returns the style of the receiver which is a bitwise OR of
 * one or more of the <code>SWT</code> constants NORMAL, BOLD
 * and ITALIC.
 *
 * @return the style of this <code>FontData</code>
 *
 * @see #setStyle
 */
public int getStyle() {
	return style;
}

/**
 * Returns an integer hash code for the receiver. Any two
 * objects that return <code>true</code> when passed to
 * <code>equals</code> must return the same value for this
 * method.
 *
 * @return the receiver's hash
 *
 * @see #equals
 */
public int hashCode () {
	return name.hashCode() ^ height ^ style;
}


	void parseFontDescriptor(String string) {
		int start = 0;
		int end = string.indexOf('|');
		if (end == -1)
			SWT.error(SWT.ERROR_INVALID_ARGUMENT);
		String version1 = string.substring(start, end);
		try {
			if (Integer.parseInt(version1) != 1)
				SWT.error(SWT.ERROR_INVALID_ARGUMENT);
		} catch (NumberFormatException e) {
			SWT.error(SWT.ERROR_INVALID_ARGUMENT);
		}

		start = end + 1;
		end = string.indexOf('|', start);
		if (end == -1)
			SWT.error(SWT.ERROR_INVALID_ARGUMENT);
		String name = string.substring(start, end);

		start = end + 1;
		end = string.indexOf('|', start);
		if (end == -1)
			SWT.error(SWT.ERROR_INVALID_ARGUMENT);
		int height = 0;
		try {
			height = Integer.parseInt(string.substring(start, end));
		} catch (NumberFormatException e) {
			SWT.error(SWT.ERROR_INVALID_ARGUMENT);
		}

		start = end + 1;
		end = string.indexOf('|', start);
		if (end == -1)
			SWT.error(SWT.ERROR_INVALID_ARGUMENT);
		int style = 0;
		try {
			style = Integer.parseInt(string.substring(start, end));
		} catch (NumberFormatException e) {
			SWT.error(SWT.ERROR_INVALID_ARGUMENT);
		}

		start = end + 1;
		end = string.indexOf('|', start);
		setName(name);
		setHeight(height);
		setStyle(style);
		if (end == -1)
			return;
		String platform = string.substring(start, end);

		start = end + 1;
		end = string.indexOf('|', start);
		if (end == -1)
			return;
		String version2 = string.substring(start, end);

		if (platform.trim().toUpperCase().equals("QT") && version2.equals("1")) {
			try {
				extraFontData = new QtSupplementaryFontData();
				start = end + 1;
				end = string.indexOf('|', start);
				if (end == -1)
					return;
				extraFontData.underline = Integer.parseInt(string.substring(
						start, end));
				start = end + 1;
				end = string.indexOf('|', start);
				if (end == -1)
					return;
				extraFontData.overline = Integer.parseInt(string.substring(
						start, end));
				start = end + 1;
				end = string.indexOf('|', start);
				if (end == -1)
					return;
				extraFontData.strikeOut = Integer.parseInt(string.substring(
						start, end));
				start = end + 1;
				end = string.indexOf('|', start);
				if (end == -1)
					return;
				int stretch = Integer.parseInt(string.substring(start, end));
				if (stretch > 0) {
					if (stretch > 4000) {
						stretch = 4000;
					}
					extraFontData.stretch = stretch;
				}
				start = end + 1;
				end = string.indexOf('|', start);
				if (end == -1) {
					extraFontData = null;
					return;
				}
				extraFontData.fixedPitch = Integer.parseInt(string.substring(
						start, end));
				start = end + 1;
				end = string.indexOf('|', start);
				if (end == -1)
					return;
				int fontStyle = Integer.parseInt(string.substring(start, end));
				if (fontStyle == OS.QFONT_STYLE_NORMAL
						|| fontStyle == OS.QFONT_STYLE_ITALIC
						|| fontStyle == OS.QFONT_STYLE_OBLIQUE) {
					extraFontData.style = fontStyle;
				}
				start = end + 1;
				end = string.indexOf('|', start);
				if (end == -1)
					return;
				int weight = Integer.parseInt(string.substring(start, end));
				if (weight > -1) {
					if (weight > 99) {
						weight = 99;
					}
					extraFontData.weight = weight;
				}
				start = end + 1;
				end = string.indexOf('|', start);
				if (end == -1)
					return;
				int styleStrategy = Integer.parseInt(string.substring(start,
						end));
				int[] strategies = new int[] {
						OS.QFONT_STYLESTRATEGY_PREFERDEFALUT,
						OS.QFONT_STYLESTRATEGY_PREFERBITMAP,
						OS.QFONT_STYLESTRATEGY_PREFERDEVICE,
						OS.QFONT_STYLESTRATEGY_PREFEROUTLINE,
						OS.QFONT_STYLESTRATEGY_FORCEOUTLINE,
						OS.QFONT_STYLESTRATEGY_NOANTIALIAS,
						OS.QFONT_STYLESTRATEGY_PREFERANTIALIAS,
						OS.QFONT_STYLESTRATEGY_OPENGLCOMPATIABLE,
						OS.QFONT_STYLESTRATEGY_NOFONTMERGING };
				boolean isValidStrategy = (styleStrategy == OS.QFONT_STYLESTRATEGY_PREFERQUALITY || styleStrategy == OS.QFONT_STYLESTRATEGY_PREFERMATCH) ? true
						: false;
				for (int i = 0; i < strategies.length; i++) {
					if ((styleStrategy & strategies[i]) == 0) {
						continue;
					} else {
						if (styleStrategy == strategies[i]
								|| styleStrategy == (strategies[i] | OS.QFONT_STYLESTRATEGY_PREFERQUALITY)
								|| styleStrategy == (strategies[i] | OS.QFONT_STYLESTRATEGY_PREFERMATCH)) {
							isValidStrategy = true;
						}
						break;
					}
				}
				if (isValidStrategy) {
					extraFontData.styleStrategy = styleStrategy;
				}
			} catch (NumberFormatException e) {
				extraFontData = null;
			}
		} else if (platform.trim().toUpperCase().equals("X11")
				&& OS.windowServer == OS.WS_X11) {
			start = end + 1;
			end = string.indexOf('|', start);
			if (end == -1)
				return;
			xlfd = string.substring(start, end);
			if (xlfd.length() < 1) {
				xlfd = null;
			}
		}

	}


/**
 * Sets the height of the receiver. The parameter is
 * specified in terms of points, where a point is one
 * seventy-second of an inch.
 *
 * @param height the height of the <code>FontData</code>
 *
 * @exception IllegalArgumentException <ul>
 *    <li>ERROR_INVALID_ARGUMENT - if the height is negative</li>
 * </ul>
 *
 * @see #getHeight
 */
public void setHeight(int height) {
	if (height < 0) SWT.error(SWT.ERROR_INVALID_ARGUMENT);
	this.height = height;
}

/**
 * Sets the locale of the receiver.
 * <p>
 * The locale determines which platform character set this
 * font is going to use. Widgets and graphics operations that
 * use this font will convert UNICODE strings to the platform
 * character set of the specified locale.
 * </p>
 * <p>
 * On platforms where there are multiple character sets for a
 * given language/country locale, the variant portion of the
 * locale will determine the character set.
 * </p>
 *
 * @param locale the <code>String</code> representing a Locale object
 * @see java.util.Locale#toString
 */
public void setLocale(String locale) {
	lang = country = variant = null;
	if (locale != null) {
		char sep = '_';
		int length = locale.length();
		int firstSep, secondSep;

		firstSep = locale.indexOf(sep);
		if (firstSep == -1) {
			firstSep = secondSep = length;
		} else {
			secondSep = locale.indexOf(sep, firstSep + 1);
			if (secondSep == -1) secondSep = length;
		}
		if (firstSep > 0) lang = locale.substring(0, firstSep);
		if (secondSep > firstSep + 1) country = locale.substring(firstSep + 1, secondSep);
		if (length > secondSep + 1) variant = locale.substring(secondSep + 1);
	}
}

/**
 * Sets the name of the receiver.
 * <p>
 * Some platforms support font foundries. On these platforms, the name
 * of the font specified in setName() may have one of the following forms:
 * <ol>
 * <li>a face name (for example, "courier")</li>
 * <li>a foundry followed by a dash ("-") followed by a face name (for example, "adobe-courier")</li>
 * </ol>
 * In either case, the name returned from getName() will include the
 * foundry.
 * </p>
 * <p>
 * On platforms that do not support font foundries, only the face name
 * (for example, "courier") is used in <code>setName()</code> and
 * <code>getName()</code>.
 * </p>
 *
 * @param name the name of the font data (must not be null)
 * @exception IllegalArgumentException <ul>
 *    <li>ERROR_NULL_ARGUMENT - when the font name is null</li>
 * </ul>
 *
 * @see #getName
 */
public void setName(String name) {
	if (name == null) SWT.error(SWT.ERROR_NULL_ARGUMENT);
	this.name = name;
}

/**
 * Sets the style of the receiver to the argument which must
 * be a bitwise OR of one or more of the <code>SWT</code>
 * constants NORMAL, BOLD and ITALIC.  All other style bits are
 * ignored.
 *
 * @param style the new style for this <code>FontData</code>
 *
 * @see #getStyle
 */
public void setStyle(int style) {
	this.style = style;
}

/**
 * Returns a string representation of the receiver which is suitable
 * for constructing an equivalent instance using the
 * <code>FontData(String)</code> constructor.
 *
 * @return a string representation of the FontData
 *
 * @see FontData
 */
public String toString() {
	StringBuffer buffer = new StringBuffer();
		buffer.append("1|");
		buffer.append(getName());
		buffer.append("|");
		buffer.append(getHeight());
		buffer.append("|");
		buffer.append(getStyle());
		buffer.append("|");
		if (xlfd != null) {
			buffer.append("X11|1|");
			buffer.append(xlfd);
			buffer.append("|");
		} else if (extraFontData != null) {
			buffer.append("QT|1|");
			buffer.append(extraFontData.overline);
			buffer.append("|");
			buffer.append(extraFontData.underline);
			buffer.append("|");
			buffer.append(extraFontData.strikeOut);
			buffer.append("|");
			buffer.append(extraFontData.stretch);
			buffer.append("|");
			buffer.append(extraFontData.fixedPitch);
			buffer.append("|");
			buffer.append(extraFontData.style);
			buffer.append("|");
			buffer.append(extraFontData.weight);
			buffer.append("|");
			buffer.append(extraFontData.styleStrategy);
			buffer.append("|");
		}
		return buffer.toString();
	}

	void updateFontData() {
		if (extraFontData == null && xlfd == null)
			return;

		boolean italic = (style & SWT.ITALIC) != 0;
		int weight = OS.QT_FONTNORMAL;
		if ((style & SWT.BOLD) != 0) {
			weight = OS.QT_FONTBOLD;
		}
		int fontHandle = OS.QFont_new(name, height, weight, italic);

		if (xlfd != null) {
			OS.QFont_setRawName(fontHandle, xlfd);

		} else {
			if (extraFontData.underline > -1) {
				OS.QFont_setUnderline(fontHandle,
						extraFontData.underline > 0 ? true : false);
			}
			if (extraFontData.overline > -1) {
				OS.QFont_setOverline(fontHandle,
						extraFontData.overline > 0 ? true : false);
			}
			if (extraFontData.strikeOut > -1) {
				OS.QFont_setStrikeOut(fontHandle,
						extraFontData.strikeOut > 0 ? true : false);
			}
			if (extraFontData.stretch > -1) {
				OS.QFont_setStretch(fontHandle, extraFontData.stretch);
			}
			if (extraFontData.fixedPitch > -1) {
				OS.QFont_setFixedPitch(fontHandle,
						extraFontData.fixedPitch > 0 ? true : false);
			}
			if (extraFontData.style > -1) {
				OS.QFont_setStyle(fontHandle, extraFontData.style);
			}
			if (extraFontData.weight > -1) {
				OS.QFont_setWeight(fontHandle, extraFontData.weight);
			}
			if (extraFontData.styleStrategy > -1) {
				OS.QFont_setStyleStrategy(fontHandle,
						extraFontData.styleStrategy);
			}
		}

		name = OS.QFont_family(fontHandle);
		height = OS.QFont_pointSize(fontHandle);
		weight = OS.QFont_weight(fontHandle);
		italic = OS.QFont_italic(fontHandle);
		style = SWT.NORMAL;
		if (weight > OS.QT_FONTNORMAL)
			style |= SWT.BOLD;
		if (italic == true)
			style |= SWT.ITALIC;
		if (extraFontData != null) {
			extraFontData.underline = OS.QFont_underline(fontHandle) == true ? 1
					: 0;
			extraFontData.overline = OS.QFont_overline(fontHandle) == true ? 1
					: 0;
			extraFontData.strikeOut = OS.QFont_strikeOut(fontHandle) == true ? 1
					: 0;
			extraFontData.stretch = OS.QFont_stretch(fontHandle);
			extraFontData.fixedPitch = OS.QFont_fixedPitch(fontHandle) == true ? 1
					: 0;
			extraFontData.style = OS.QFont_style(fontHandle);
			extraFontData.weight = OS.QFont_weight(fontHandle);
			extraFontData.styleStrategy = OS.QFont_styleStrategy(fontHandle);
			OS.QFont_delete(fontHandle);
		}
	}
}
